/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Runtime.Serialization;
using Borland.Eco.UmlRt;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Services;
using Borland.Eco.Subscription;
using Borland.Eco.Globalization;

namespace Borland.Eco.Ocl.Support
{

	public enum OclSortDirection { ascending, descending };

	public interface IOclTypeFactory
	{
		IOclType TypeForClassifier(IClassifier classifier);
		IOclType TypeType{get;}
	}

	public interface IOclType
	{
		string Name();
		bool IsListType();
		bool IsObjectListType();
		bool IsObjectType();
		bool IsPrimitiveType();
		bool IsEnumType();
		bool IsSimpleType();
		bool IsSameType(IOclType oclType);
		bool ConformsTo(IOclType oclType);
		bool HasUmlType();
		IOclType BagType();
		IOclType SetType();
		IOclType SequenceType();
		IOclType OrderedSetType();
		IOclType CollectionType { get; }
		IOclType ElementType { get; }
		IOclType LeastCommonType(IOclType oclType);
		IClassifier UMLType { get; }
	}

	public interface IOclOperationSupport
	{
		// predefined types
		IEcoTypeSystem Model { get; }
		IOclType BooleanType { get; }
		IOclType ByteType { get; }
		IOclType SByteType { get; }
		IOclType Int64Type { get; }
		IOclType Int16Type { get; }
		IOclType UInt16Type { get; }
		IOclType IntegerType { get; }
		IOclType RealType { get; }
		IOclType DecimalType { get; }
		IOclType TimeSpanType { get; }
		IOclType DateTimeType { get; }
		IOclType StringType { get; }
		IOclType ListType { get; }
		IOclType ObjectType { get; }
		IOclType TypeType { get; }
		IOclType AnyType { get; }
		IOclType OclSortDirection { get; }
		IOclType EnumLiteralType { get; }
		IOclType NumericListType { get; }
		IOclType DecimalListType { get; }
		IOclType TimeSpanListType { get; }
		IOclType DateTimeListType { get; }
		IOclType StringListType { get; }
		IOclType ConstraintListType { get; }
		IOclType ObjectListType { get; }

		// services
		IVariableFactoryService VariableFactory { get; }
		IExternalIdService ExternalIdService { get; }
		IOclService OclService { get; }
		IExtentService ExtentService { get; }
		IVersionService VersionService { get; }
		IObjectFactoryService ObjectFactory { get; }
		IUndoService UndoService { get; }

		// constructors
		IElement CreateNewConstant(IOclType type, object initialValue);
		IElement CreateNewVariable(IOclType type);
		IElement CreateNewSimpleVariable(ISimpleType type);

		void MakeNewNumeric(IOclValue node, double value);
		void MakeNewDecimal(IOclValue node, decimal value);
		void MakeNewTimeSpan(IOclValue node, TimeSpan value);
		void MakeNewDateTime(IOclValue node, DateTime value);
		void MakeNewBoolean(IOclValue node, bool value);
		void MakeNewInteger(IOclValue node, int value);
		void MakeNewInt64(IOclValue node, long value);
		void MakeNewString(IOclValue node, string value);

		// accessors
		bool GetAsBoolean(IOclValue node);
		int GetAsInteger(IOclValue node);
		long GetAsInt64(IOclValue Node);
		DateTime GetAsDateTime(IOclValue node);
		TimeSpan GetAsTimeSpan(IOclValue node);
		TimeSpan GetAsTimeSpan(IElement element);
		double GetAsNumeric(IOclValue node);
		double GetAsNumeric(IElement element);
		decimal GetAsDecimal(IOclValue node);
		decimal GetAsDecimal(IElement element);
		string GetAsString(IOclValue node);
		IElementCollection GetAsList(IOclValue node);
		IClassifier GetAsType(IOclValue bode);

		// other support functions
		void TransferOrClone(IOclValue source, IOclValue dest);
		object ConvertElement(IOclType toType, IElement Element);
		IOclType GetOclTypeFor(IClassifier classifier);
	}

	public interface IOclValue
	{
		IElement Element { get; }
		IOclType OclType { get; }
		void SetOwnedElement(IElement value);
		void SetReferenceElement(IElement value);
		void SetReferenceClassifier(IClassifier value);
		void SetDefaultValue(IOclOperationSupport support);
		bool IsTypeNode { get; }
		bool OwnsValue { get; }
//        int Pos { get; }
		IClassifier ClassifierValue { get; }
		void TransferValue(IOclValue node);
	}

	public interface IOclNamedEnumLiteral: IOclValue
	{
		string Name { get; }
	}

	public interface IOclOperationParameters
	{
		IOclValue[] Values { get; }
		IOclValue Result { get; }
		SubscriberPair SubscriberBlock { get; }
	}

	/// <summary>
	/// This interface is implemented by all operations that 
	/// is installed in the Ocl evaluator (or Eco Action Language
	/// executor
	/// There is a handy base implementation of this interface in 
	/// Borland.Eco.Ocl.Support.OclOperationBase that can be subclassed
	/// </summary>
	public interface IOclOperation
	{
		/// <summary>
		/// The type of the formal parameters. The sematic check will use this to determine if the actual parameters can be used to call this operation
		/// </summary>
		IOclType[] FormalParameters { get; }
		/// <summary>
		/// If this value is non-zero, it means that the last x parameters can be repeated when the operation is called
		/// </summary>
		int RepeatParameters { get; }
		/// <summary>
		/// The return type of the operation. Requires the DeduceMethod to be set to OclResultTypeDeduceMethod.ExplicitType
		/// </summary>
		IOclType ResultType { get; }
		/// <summary>
		/// Indicates how the resulttype is calculated based on the types of the actual parameters.
		/// </summary>
		OclResultTypeDeduceMethod DeduceMethod { get; }
		/// <summary>
		/// The name of the operation
		/// </summary>
		string Name { get; }
		/// <summary>
		/// if true, it means that the operation must be prefixed with a "."
		/// </summary>
		bool IsDotNotation { get; }
		/// <summary>
		/// True for "normal" operations. False for "infix" operations supported by the parser, such as "and", "+", etc.
		/// </summary>
		bool IsPostFix { get; }
		/// <summary>
		/// this should be true if the operation has side effects on the source-parameter (such as "clear")
		/// </summary>
		bool ModifiesSource { get; }
		/// <summary>
		/// Determines if the result of the operation is readonly based on the readonlyness of the parameters
		/// </summary>
		bool ResultIsReadOnly(bool[] args);
		/// <summary>
		/// Determines if the operation requires the source and the first parameter to be able to conform to each other one way or the other
		/// </summary>
		bool ArgsNeedCommonType { get; }
		/// <summary>
		/// If this is true, then the operation can be applied to the type directly (corresponds to c# "static" or Delphi "class method"
		/// </summary>
		bool IsTypeOperation { get; }
		/// <summary>
		/// if the operation has custom semantic validation, it can be implemented here (OclOperationBase has an empty implemenation of this method)
		/// </summary>
		void SemanticCheck(IOclOperationParameters oclParameters);
		/// <summary>
		/// This method performs the actual evaluation (/execution) of the operation
		/// </summary>
		void Evaluate(IOclOperationParameters oclParameters);
		void Initialize(IOclOperationSupport support);
	}

	public enum OclResultTypeDeduceMethod
	{
		ExplicitType,
		LoopVariableType,
		TypeOfSource,
		ListElementTypeOfSource,
		TypeOfFirstParam,
		TypeOfSecondParam,
		LeastCommonType_SourceAndFirstParam,
		LeastCommonType_FirstAndSecondParam,
		FirstParamAsSequence,
		ObjectListFromSourceType,
		MetaType,
		TypeCast,
		SourceAsBag,
		SourceAsSet,
		SourceAsSequence,
		SourceAsOrderedSet,
		ObjectListFromFirstParamType,
		Boolean_TypesMustShareDomain,
		ObjectFromSourceType
	}

	public class ProxyElement: IElement
	{
		public ProxyElement(IElement el) { this.el = el; }
		private IElement el;	
		IElement IElement.CloneValue(bool readOnly) { return el.CloneValue(readOnly); }
		bool IElement.IsEqual(IElement element) { return el.IsEqual(element); }
		int IElement.GetIsEqualHashCode() { return el.GetIsEqualHashCode(); }
		void IElement.Assign(IElement element) { el.Assign(element); }
		void IElement.SubscribeToValue(ISubscriber subscriber) { el.SubscribeToValue(subscriber); }

		ContentType IElement.ContentType { get { return el.ContentType; }}
		IElementCollection IElement.GetAsCollection() { return el.GetAsCollection();}
		bool IElement.Mutable { get { return el.Mutable; }}
		bool IElement.ReadOnly { get { return el.ReadOnly; }}
		bool IElement.IsProperty { get { return el.IsProperty; }}
		IClassifier IElement.UmlType { get {return el.UmlType; }}
		System.Object IElement.AsObject { get {return el.AsObject; } set { el.AsObject = value; } }
	}
	
	public class SortElement: ProxyElement
	{
		public SortElement(IElement el, OclSortDirection direction): base(el)
		{ 
		    this.direction = direction;
		}
		private OclSortDirection direction; 	
		public OclSortDirection Direction { get { return direction; }}
	}


	[AttributeUsage(AttributeTargets.Class)]
	public sealed class EcoOclOperationAttribute: Attribute
	{
		private System.Type m_Operation;
		private bool m_IsQuery;

		public System.Type Operation
		{
			get { return m_Operation; }
		}

		public bool IsQuery
		{
			get { return m_IsQuery; }
		}

		public EcoOclOperationAttribute(System.Type operation, bool isQuery): base()
		{
			m_Operation = operation;
			m_IsQuery = isQuery;
		}
	}
	[AttributeUsage(AttributeTargets.Class)]
	public sealed class EcoOclOperationHelpAttribute: Attribute
	{
		private string m_HelpString;
		public EcoOclOperationHelpAttribute(string helpString)
		{
			m_HelpString = helpString;
		}
		public string HelpString { get { return m_HelpString; } }
	}
	[AttributeUsage(AttributeTargets.Class)]
	public sealed class EcoOclOperationResourcedHelpAttribute: Attribute
	{
		private string m_HelpResource;
		public EcoOclOperationResourcedHelpAttribute(string helpResource)
		{
			m_HelpResource = helpResource;
		}
		public string HelpString(object requestor)
		{
			return ResManager.GetString(requestor, m_HelpResource);
		}
		public string HelpResource { get { return m_HelpResource; } }
	}
}

namespace Borland.Eco.Exceptions
{
	[Serializable]
	public abstract class EcoOclErrorException: EcoException
	{
		protected EcoOclErrorException(): base() {}
		protected EcoOclErrorException(string message): base(message) {}
		protected EcoOclErrorException(string message, Exception inner): base(message, inner) {}
#if !CF
		protected EcoOclErrorException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
//		protected EcoOclErrorException(string message): base(message) {}
		public abstract int Position {get;}
		public abstract string Expression{get;}
		public abstract bool InternalError {get;}
	}
}
